#include <sys/types.h>
#include <sys/socket.h>

#include <endian.h>
#include <errno.h>
#include <inttypes.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>

#include "mu.h"


#define REQUEST_SIZE 8
#define RESPONSE_SIZE 4
#define MAX_MESSAGE_SIZE 8


static int
client_create(const char *srv_ip, const char *srv_port)
{
    struct sockaddr_in sa;
    int sk;
    int err;

    sk = socket(AF_INET, SOCK_STREAM, 0);
    if (sk == -1)
        mu_die_errno(errno, "socket");

    mu_init_sockaddr_in(&sa, srv_ip, srv_port);
    err = connect(sk, (struct sockaddr *)&sa, sizeof(sa));
    if (err == -1)
        mu_die_errno(errno, "connect");

    return sk;
}


static uint32_t
do_add_rpc(int sk, uint32_t x, uint32_t y)
{
    uint8_t msg[MAX_MESSAGE_SIZE] = { 0 };
    int err;
    size_t total = 0;
    uint32_t tmp, res;

    /* serialize request */
    x = htobe32(x);
    memcpy(msg, &x, sizeof(x));
    y = htobe32(y);
    memcpy(msg + sizeof(x), &y, sizeof(y));

    /* send request */
    err = mu_write_n(sk, msg, REQUEST_SIZE, &total);
    if (err < 0)
        mu_die_errno(-err, "send failed");

    /* receive response */
    err = mu_read_n(sk, msg, RESPONSE_SIZE, &total);
    if (err < 0)
        mu_die_errno(-err, "failed to receive response");
    else if (total != RESPONSE_SIZE)
        mu_die("server disconnected");

    /* deserialize response */
    tmp = *((uint32_t *)msg);
    res = be32toh(tmp);

    return res;
}


int 
main(int argc,char *argv[])
{
    int err;
    uint32_t x, y, res;
    int sk;

    if (argc != 5)
        mu_die("Usage: %s IP PORT X Y", argv[0]);

    err = mu_str_to_u32(argv[3], 10, &x);
    if (err != 0)
        mu_die("invalid x");

    err = mu_str_to_u32(argv[4], 10, &y);
    if (err != 0)
        mu_die("invalid y");

    sk = client_create(argv[1], argv[2]);

    res = do_add_rpc(sk, x, y);

    printf("%" PRIu32 "\n", res);

    close(sk);

    return 0;
}
