#include <sys/types.h>
#include <sys/socket.h>

#include <endian.h>
#include <errno.h>
#include <inttypes.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>

#include "mu.h"


#define BACKLOG 5
#define REQUEST_SIZE 8
#define RESPONSE_SIZE 4
#define MAX_MESSAGE_SIZE 8


static void
serve_forever(int sk)
{
    struct sockaddr_in addr;
    socklen_t addr_size;
    int conn;
    char peer_str[MU_LIMITS_MAX_INET_STR_SIZE] = { 0 };
    int err = 0;
    size_t total;
    uint8_t msg[REQUEST_SIZE] = { 0 };
    uint32_t x, y, res;

    while (1) {
        addr_size = sizeof(addr);
        conn = accept(sk, (struct sockaddr *)&addr, &addr_size);
        if (conn == -1)
            mu_die_errno(errno, "accept");

        mu_sockaddr_in_to_str(&addr, peer_str, sizeof(peer_str));
        mu_pr_debug("%s: connected", peer_str);

        /* receive request */
        err = mu_read_n(conn, msg, sizeof(msg), &total);
        if (err < 0) {
            mu_stderr_errno(-err, "%s: error handling TCP request", peer_str);
            goto request_done;
        } else if (total != sizeof(msg)) {
            mu_stderr("%s: disconnected: failed to receive complete request", peer_str);
            goto request_done;
        }

        /* deserialize request */
        memcpy(&x, msg, sizeof(x));
        x = be32toh(x);
        memcpy(&y, msg + sizeof(x), sizeof(y));
        y = be32toh(y);

        /* process request */
        res = x + y;

        mu_pr_debug("%s: %" PRIu32 " + %" PRIu32 " = %" PRIu32,
                peer_str, x, y, res);

        /* serialize response */
        res = htobe32(res);
        memcpy(msg, &res, sizeof(res));

        /* send response */
        err = mu_write_n(conn, msg, RESPONSE_SIZE, &total);
        if (err < 0)
            mu_stderr_errno(-err, "%s: TCP send failed", peer_str);

request_done:
        close(conn);
    }
}


static int
tcp_server_create(const char *ip, const char *port)
{
    struct sockaddr_in sa;
    int sk;
    int err;

    sk = socket(AF_INET, SOCK_STREAM, 0);
    if (sk == -1)
        mu_die_errno(errno, "socket");

    mu_reuseaddr(sk);

    mu_init_sockaddr_in(&sa, ip, port);
    err = bind(sk, (struct sockaddr *)&sa, sizeof(sa));
    if (err == -1)
        mu_die_errno(errno, "bind");

    err = listen(sk, BACKLOG);
    if (err == -1)
        mu_die_errno(errno, "listen");

    return sk;
}


int
main(int argc, char *argv[])
{
    int sk;

    if (argc != 2)
        mu_die("Usage: %s PORT", argv[0]);

    sk = tcp_server_create("0.0.0.0", argv[1]);

    serve_forever(sk);

    return 0;
}
