#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include "mu.h"

/* Return 0 on success, -1 on error */
static int
read_range(const char *path, long start, long end)
{
    int ret = 0;
    int fd;
    off_t off;
    char buf[4] = { 0 };
    long left = end - start;
    size_t want = 0;
    ssize_t n = 0;

    fd = open(path, O_RDONLY);
    if (fd == -1) {
        ret = -1;
        mu_stderr_errno(errno, "%s", path);
        goto out;
    }

    off = lseek(fd, start, SEEK_SET);
    if (off == -1) {
        ret = -1;
        mu_stderr_errno(errno, "lseek");
        goto out;
    }

    want = MU_MIN(sizeof(buf), (size_t)left);
    while (want) {
        n = read(fd, buf, want);
        if (n == -1) {
            ret = -1;
            mu_stderr_errno(errno, "error reading %zu bytes from \"%s\"", want, path);
            goto out;
        }
        if (n == 0)
            break;

        ret = mu_write_n(STDOUT_FILENO, buf, (size_t)n, NULL);
        if (ret != 0) {
            mu_stderr_errno(-ret, "error writing to stdout");
            ret = -1;
            goto out;
        }

        left -= n;
        want = MU_MIN((size_t)left, sizeof(buf));
    }

out:
    if (fd != -1)
        close(fd);

    return ret;
}


int
main(int argc, char *argv[])
{
    int ret;
    long start, end;

    if (argc != 4)
        mu_die("Usage: %s PATH START END", argv[0]);

    ret = mu_str_to_long(argv[2], 10, &start);
    if (ret != 0)
        mu_die_errno(-ret, "bad START value: \"%s\"", argv[2]);

    ret = mu_str_to_long(argv[3], 10, &end);
    if (ret != 0)
        mu_die_errno(-ret, "bad END value: \"%s\"", argv[3]);

    ret = read_range(argv[1], start, end);
    
    return ret == 0 ? 0 : 1;
}
