use clap::Parser;
use std::fs::File;
use std::io::{self, BufRead, BufReader};
use std::process;

#[derive(Parser, Debug)]
#[command(name = "cat", about = "A simple version of cat")]
struct Args {
    /// Number all output lines (-n, --number)
    #[arg(short = 'n', long = "number")]
    number: bool,

    /// Files to read
    #[arg(value_name = "FILE")]
    files: Vec<String>,
}

fn cat_file(path: &str, number: bool, line_no: &mut usize) -> io::Result<()> {
    let file = File::open(path)?;
    let reader = BufReader::new(file);
    for line in reader.lines() {
        let line = line?;
        if number {
            *line_no += 1;
            println!("{:>6}\t{}", *line_no, line);
        } else {
            println!("{}", line);
        }
    }
    Ok(())
}

fn main() {
    let args = Args::parse();

    let mut line_no: usize = 0;
    let mut had_error = false;

    for path in &args.files {
        if let Err(e) = cat_file(path, args.number, &mut line_no) {
            eprintln!("cat: {}: {}", path, e);
            had_error = true;
        }
    }

    if had_error {
        process::exit(1);
    }
}
