#include <sys/types.h>
#include <sys/signalfd.h>

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#include "mu.h"


static const char *
strcode(int si_code)
{
    switch (si_code) {
    case SI_USER:       return "sent by user";
    case SI_KERNEL:     return "sent by kernel";
    case SI_QUEUE:      return "sent by sigqueue";
    case SI_TIMER:      return "POSIX timer expired";
    case SI_MESGQ:      return "POSIX message queue state chnaged";
    case SI_ASYNCIO:    return "AIO completed";
    case SI_SIGIO:      return "Queued SIGIO";
    case SI_TKILL:      return "tkill";

    /* TODO: others: see `man sigaction` */

    default:            return "unknown";
    }
}


int
main(int argc, char *argv[])
{
    int ret;
    int secs;
    int fd;
    ssize_t n;
    struct signalfd_siginfo ssi;
    sigset_t all_sigs;

    printf("pid: %d\n", getpid());

    /* block all signals (except SIGKILL and SIGSTOP) */
    sigfillset(&all_sigs);
    if (sigprocmask(SIG_SETMASK, &all_sigs, NULL) == -1)
        mu_die_errno(errno, "sigprocmask");

    if (argc > 1) {
        ret = mu_str_to_int(argv[1], 10, &secs);
        if (ret != 0)
            mu_die_errno(-ret, "invalid value for delay: \"%s\"", argv[1]);
        if (secs < 0)
            mu_die_errno(-ret, "delay must be non-negative");

        printf("delaying for %d seconds\n", secs);
        sleep(secs);
    }

    fd = signalfd(-1, &all_sigs, 0);
    if (fd == -1)
        mu_die_errno(errno, "signalfd");

    while (1) {
        /* wait until a signal is pending */
        n = read(fd, &ssi, sizeof(ssi));
        if (n == -1)
            mu_die_errno(errno, "read");
        if (n != sizeof(ssi))
            mu_panic("unexpected partial read of %zd / %zu bytes", n, sizeof(ssi));

        if (ssi.ssi_signo == SIGINT || ssi.ssi_signo == SIGTERM)
            break;

        printf("got: {sig %d (%s), code %d (%s)",
                ssi.ssi_signo, strsignal(ssi.ssi_signo), ssi.ssi_code,
                strcode(ssi.ssi_code));
        if (ssi.ssi_code == SI_QUEUE)
            printf(", value %d\n", ssi.ssi_int);
        puts("}");

    }

    close(fd);
    return 0;
}
