#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "mu.h"


static volatile sig_atomic_t g_got_sigquit = 0;


static void
print_sigset(const char *msg, const sigset_t *set)
{
    int sig, n = 0;
   
    puts(msg);
    for (sig = 1; sig < NSIG; sig++) {
        if (sigismember(set, sig)) {
            printf("\t%d (%s)\n", sig, strsignal(sig));
            n++;
        }
    }

    if (n == 0)
        puts("\t<none>\n");
}


static void
print_signal_mask(const char *msg)
{
    sigset_t cur_mask;

    if (sigprocmask(SIG_BLOCK, NULL, &cur_mask) == -1)
        mu_panic_errno(errno, "sigprocmask");

    print_sigset(msg, &cur_mask);
}


static void
print_pending_signals(const char *msg)
{
    sigset_t pending;

    if (sigpending(&pending) == -1)
        mu_panic("sigpending");

    print_sigset(msg, &pending);
}


/* UNSAFE: this handler uses a non-async-signal-safe function (printf) */
static void
handler(int sig, siginfo_t *info, void *ucontext)
{
    MU_UNUSED(info);
    MU_UNUSED(ucontext);

    printf("[caught signal %d (%s)]\n", sig, strsignal(sig));

    if (sig == SIGQUIT)
        g_got_sigquit = 1;
}


int
main(void)
{
    struct sigaction act;
    sigset_t orig_mask, block_mask;
    int i;
    time_t start;

    /* print original signal mask */
    print_signal_mask("original signal mask");

    /* block SIGINT and SIGQUIT */
    sigemptyset(&block_mask);
    sigaddset(&block_mask, SIGINT);
    sigaddset(&block_mask, SIGQUIT);
    if (sigprocmask(SIG_BLOCK, &block_mask, &orig_mask) == -1)
        mu_die_errno(errno, "sigprocmask(SIG_BLOCK)");

    /* establish a handler for SIGINT and SIGQUIT */
    memset(&act, 0x00, sizeof(act));
    sigemptyset(&act.sa_mask);
    act.sa_sigaction = handler;

    if (sigaction(SIGINT, &act, NULL) == -1)
        mu_die_errno(errno, "sigaction(SIGINT)");
    if (sigaction(SIGQUIT, &act, NULL) == -1)
        mu_die_errno(errno, "sigaction(SIGINT)");


    /* processing loop */
    for (i = 0; !g_got_sigquit; i++) {
        printf("loop %d\n", i);

        /* simulate critical section by busy looping for a few seconds */
        print_signal_mask("  signal mask during critical section");
        for (start = time(NULL); time(NULL) < start + 4; ) {
            ;
        }

        /* print signals that became pending during critical section */
        print_pending_signals("  pending signals during critical section");

        /* 
         * on entry: unblocks SIGINT and SIGQUIT; after delivery of (any)
         * signal, returns and restores the prior mask that blocked SIGINT and
         * SIGQUIT
         */
        if (sigsuspend(&orig_mask) == -1 && errno != EINTR)
            mu_die_errno(errno, "sigsuspend");
    }
    

    /* restore old mask (i.e., unblock SIGINT and SIGQUIT) */
    if (sigprocmask(SIG_SETMASK, &orig_mask, NULL) == -1)
        mu_die_errno(errno, "sigprocmask(SIG_SETMASK)");

    print_signal_mask("restored signal mask back to original");

    /* do other processing */

    return 0;
}
