#include <sys/types.h>
#include <sys/socket.h>

#include <assert.h>
#include <endian.h>
#include <errno.h>
#include <inttypes.h>
#include <pthread.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>

#include "mu.h"


#define BACKLOG 10
#define REQUEST_SIZE 8
#define RESPONSE_SIZE 4
#define MAX_MESSAGE_SIZE 8


struct client {
    int conn;
    char peer_str[MU_LIMITS_MAX_INET_STR_SIZE];
};


static struct client *
client_new(int conn, const char *peer_str)
{
    MU_NEW(client, c);
    size_t len;

    c->conn = conn;
    len = mu_strlcpy(c->peer_str, peer_str, sizeof(c->peer_str));
    assert(len < sizeof(c->peer_str));

    return c;
}


static void
client_free(struct client *c)
{
    free(c);
}


static void *
handle_client(void * arg)
{
    int err = 0;
    size_t total;
    uint8_t msg[REQUEST_SIZE] = { 0 };
    uint32_t x, y, res;
    struct client *c = arg;

    /* receive request */
    err = mu_read_n(c->conn, msg, sizeof(msg), &total);
    if (err < 0) {
        mu_stderr_errno(-err, "%s: error handling TCP request", c->peer_str);
        goto request_done;
    } else if (total != sizeof(msg)) {
        mu_stderr("%s: disconnected: failed to receive complete header", c->peer_str);
        goto request_done;
    }

    /* deserialize request */
    memcpy(&x, msg, sizeof(x));
    x = be32toh(x);
    memcpy(&y, msg + sizeof(x), sizeof(y));
    y = be32toh(y);

    /* process request */
    res = x + y;

    mu_pr_debug("%s: %" PRIu32 " + %" PRIu32 " = %" PRIu32,
            c->peer_str, x, y, res);

    /* serialize response */
    res = htobe32(res);
    memcpy(msg, &res, sizeof(res));

    /* send response */
    err = mu_write_n(c->conn, msg, RESPONSE_SIZE, &total);
    if (err < 0)
        mu_stderr_errno(-err, "%s: TCP send failed", c->peer_str);

request_done:
        mu_pr_debug("%s: closing", c->peer_str);
        close(c->conn);
        client_free(c);

        return NULL;
}


static void
serve_forever(int sk)
{
    struct sockaddr_in addr;
    socklen_t addr_size;
    int conn;
    char peer_str[MU_LIMITS_MAX_INET_STR_SIZE] = { 0 };
    struct client *c = NULL;
    pthread_t thread;
    int err;


    while (1) {
        addr_size = sizeof(addr);
        conn = accept(sk, (struct sockaddr *)&addr, &addr_size);
        if (conn == -1)
            mu_die_errno(errno, "accept");

        mu_sockaddr_in_to_str(&addr, peer_str, sizeof(peer_str));
        mu_pr_debug("%s: connected", peer_str);

        c = client_new(conn, peer_str);
        err = pthread_create(&thread, NULL, handle_client, c);
        if (err != 0)
            mu_stderr_errno(err, "pthread_create");

        err = pthread_detach(thread);
        if (err != 0)
            mu_stderr_errno(err, "pthread_detach");
    }
}


static int
tcp_server_create(const char *ip, const char *port)
{
    struct sockaddr_in sa;
    int sk;
    int err;

    sk = socket(AF_INET, SOCK_STREAM, 0);
    if (sk == -1)
        mu_die_errno(errno, "socket");

    mu_reuseaddr(sk);

    mu_init_sockaddr_in(&sa, ip, port);
    err = bind(sk, (struct sockaddr *)&sa, sizeof(sa));
    if (err == -1)
        mu_die_errno(errno, "bind");

    err = listen(sk, BACKLOG);
    if (err == -1)
        mu_die_errno(errno, "listen");

    return sk;
}


int
main(int argc, char *argv[])
{
    int sk;

    if (argc != 2)
        mu_die("Usage: %s PORT", argv[0]);

    sk = tcp_server_create("0.0.0.0", argv[1]);

    serve_forever(sk);

    return 0;
}
