package sodoh

import (
	"fmt"
	"net"
)

// QName is the domainname to resolve
type DNSQuery struct {
	QName string `json:"qname"`
}

// WrappedKey is an RSA-OAEP encrypted AES256 key
// Ciphertext is an AES256-GCM encrypted JSON string (a marshalled DNSQuery)
type ODOHQuery struct {
	WrappedKey []byte `json:"wrappedKey"`
	Ciphertext []byte `json:"ciphertext"`
}

type RCode uint16

const (
	RCodeNOERROR RCode = iota
	RCodeNXDOMAIN
	RCodeSERVFAIL
)

func (rcode RCode) String() string {
	switch rcode {
	case RCodeNOERROR:
		return "No Error"
	case RCodeNXDOMAIN:
		return "Non-Existent Domain"
	case RCodeSERVFAIL:
		return "Server Error"
	default:
		return fmt.Sprintf("Unknown DNS Error (%d)", rcode)
	}
}

type DNSReply struct {
	RCode   RCode    `json:"rcode"`
	Answers []string `json:"answers,omitempty"`
}

// Ciphertext is an AES256-GCM encrypted JSON string (a marshalled DNSReply)
type ODOHReply struct {
	Ciphertext []byte `json:"ciphertext"`
}

// ResolveDomain returns a list of IPv4 and IPv6 addresses for domainName.
func ResolveDomain(domainName string) ([]string, error) {
	return net.LookupHost(domainName)
}
