package main

import (
	"bufio"
	"flag"
	"fmt"
	"io"
	"log"
	"net"
	"strings"

	"ucase_tcp/internal/mu"
)

const shortUsage = "Usage: server HOSTPORT"
const usage = `Usage: server [options] HOSTPORT

TCP ucase server

positional arguments:
  HOSTPORT
    The IP address and port number to listen on.

options:
  -h, -help
    Show this usage statement and exit.

examples:
$ ./client 127.0.0.1:12345
`

func printUsage() {
	fmt.Print(usage)
}

func handleConn(conn net.Conn) {
	defer func() {
		conn.Close()
		log.Printf("closed connection to %s", conn.RemoteAddr())
	}()

	scanner := bufio.NewScanner(conn)
	for scanner.Scan() {
		input := scanner.Text()
		log.Printf("got %q from %v", input, conn.RemoteAddr())
		output := strings.ToUpper(input + "\n")
		_, err := io.WriteString(conn, output)
		if err != nil {
			log.Printf("error writing to client %v: %v", conn.RemoteAddr(), err)
			return
		}
	}

	if err := scanner.Err(); err != nil {
		log.Printf("error reading from client %v: %v", conn.RemoteAddr(), err)
	}
}

func serveForever(server net.Listener) {
	for {
		conn, err := server.Accept()
		if err != nil {
			log.Print(err) // e.g., connection aborted
			continue
		}
		log.Printf("new connection from %v", conn.RemoteAddr())
		go handleConn(conn) // handle connections concurrently
	}
}

func createServer(hostport string) net.Listener {
	server, err := net.Listen("tcp", hostport)
	if err != nil {
		log.Fatal(err)
	}
	log.Printf("listening on % v", server.Addr())
	return server
}

func main() {
	log.SetPrefix("[server] ")
	log.SetFlags(log.Lmsgprefix | log.LstdFlags | log.LUTC)

	flag.Usage = printUsage
	flag.Parse()

	if flag.NArg() != 1 {
		mu.Die(shortUsage)
	}

	hostport := flag.Arg(0)

	server := createServer(hostport)
	defer server.Close()
	serveForever(server)
}
