package main

import (
	"bufio"
	"crypto/tls"
	"crypto/x509"
	"flag"
	"fmt"
	"io"
	"net"
	"os"

	"ucase_tls/internal/mu"
)

const shortUsage = "Usage: client HOSTPORT CA_CERT"
const usage = `Usage: client HOSTPORT CA_CERT

TCP ucase client

HOSTPORT
  The server's IP address and port number to connect to.

CA_CERT
  A certificate for a trusted root CA.

options:
  -h, -help
    Show this usage statement and exit.

examples:
$ ./client 127.0.0.1:12345 ca.crt
`

func printUsage() {
	fmt.Print(usage)
}

func repl(conn net.Conn) {
	defer conn.Close()

	scanner := bufio.NewScanner(os.Stdin)
	connBufReader := bufio.NewReader(conn)
	for scanner.Scan() {
		s := scanner.Text()
		_, err := io.WriteString(conn, s+"\n")
		if err != nil {
			mu.Die("error: socket write failed: %v", err)
		}

		text, err := connBufReader.ReadString('\n')
		fmt.Print(text)
		if err != nil && err != io.EOF {
			mu.Die("error: socket read failed: %v", err)
			break
		}
	}

	if err := scanner.Err(); err != nil {
		mu.Die("error: stdin read failed: %v", err)
	}
}

func createConn(hostport, caCertFile string) net.Conn {
	cert, err := os.ReadFile(caCertFile)
	if err != nil {
		mu.Die("failed to read CA cert: %v", err)
	}

	certPool := x509.NewCertPool()
	if ok := certPool.AppendCertsFromPEM(cert); !ok {
		mu.Die("failed to parse CA cert from %s", caCertFile)
	}

	config := tls.Config{RootCAs: certPool}

	conn, err := tls.Dial("tcp", hostport, &config)
	if err != nil {
		mu.Die("error: %v", err)
	}
	return conn
}

func main() {
	flag.Usage = printUsage
	flag.Parse()

	if flag.NArg() != 2 {
		mu.Die(shortUsage)
	}

	hostport := flag.Arg(0)
	caCertFile := flag.Arg(1)

	conn := createConn(hostport, caCertFile)
	repl(conn)
}
