package main

import (
	"bufio"
	"crypto/tls"
	"flag"
	"fmt"
	"io"
	"log"
	"net"
	"strings"

	"ucase_tls/internal/mu"
)

const shortUsage = "Usage: server HOSTPORT CERT KEY"
const usage = `Usage: server [options] HOSTPORT CERT KEY

TCP ucase server

positional arguments:
  HOSTPORT
    The IP address and port number to listen on.

  CERT
    The server's TLS certificate.

  KEY
    The server's TLS KEY.

options:
  -h, -help
    Show this usage statement and exit.

examples:
$ ./server 127.0.0.1:12345 server.crt server.key
`

func printUsage() {
	fmt.Print(usage)
}

func handleConn(conn net.Conn) {
	defer func() {
		conn.Close()
		log.Printf("closed connection to %s", conn.RemoteAddr())
	}()

	scanner := bufio.NewScanner(conn)
	for scanner.Scan() {
		input := scanner.Text()
		log.Printf("got %q from %v", input, conn.RemoteAddr())
		output := strings.ToUpper(input + "\n")
		_, err := io.WriteString(conn, output)
		if err != nil {
			log.Printf("error writing to client %v: %v", conn.RemoteAddr(), err)
			return
		}
	}

	if err := scanner.Err(); err != nil {
		log.Printf("error reading from client %v: %v", conn.RemoteAddr(), err)
	}
}

func serveForever(server net.Listener) {
	for {
		conn, err := server.Accept()
		if err != nil {
			log.Print(err) // e.g., connection aborted
			continue
		}
		log.Printf("new connection from %v", conn.RemoteAddr())
		go handleConn(conn) // handle connections concurrently
	}
}

func createServer(hostport, certFile, keyFile string) net.Listener {
	config := &tls.Config{}

	cert, err := tls.LoadX509KeyPair(certFile, keyFile)
	if err != nil {
		log.Fatalf("failed to load keypair: %v", err)
	}

	config.Certificates = append(config.Certificates, cert)

	server, err := tls.Listen("tcp", hostport, config)
	if err != nil {
		log.Fatal(err)
	}
	log.Printf("listening on % v", server.Addr())
	return server
}

func main() {
	log.SetPrefix("[server] ")
	log.SetFlags(log.Lmsgprefix | log.LstdFlags | log.LUTC)

	flag.Usage = printUsage
	flag.Parse()

	if flag.NArg() != 3 {
		mu.Die(shortUsage)
	}

	hostport := flag.Arg(0)
	certFile := flag.Arg(1)
	keyFile := flag.Arg(2)

	server := createServer(hostport, certFile, keyFile)
	defer server.Close()
	serveForever(server)
}
